<?php

namespace Payamito\Modules\Booking;

use Payamito_Jet_Admin;
use Payamito_Jet_User;

if (!class_exists('Callbacks')) {
    class Callbacks
    {
        protected static $_instance = null;
        public array $booking = [];
        public $data;
        public $settings;
        public $general_settings = [];
        public $admin_phones;
        public $validate = true;

        public static function get_instance()
        {
            if (is_null(self::$_instance)) {
                self::$_instance = new self();
            }
            return self::$_instance;
        }

        public function __construct()
        {
            if (function_exists('jet_abaf')) {
                add_filter('jet-booking/form-action/pre-process', [$this, 'pre_process'], 999, 3);
                add_action('jet-booking/form-action/booking-inserted', [$this, 'add']);
                add_action('payamito/jet/shutdown', [$this, 'shutdown']);
                add_filter('rest_request_before_callbacks', [$this, 'is_booking_status_change'], 999, 3);
                add_filter('rest_request_after_callbacks', [$this, 'update'], 999, 3);
                add_action('woocommerce_order_status_changed', [$this, 'update_status_on_order_update'], 9999, 1);
            }
        }

        public function update_status_on_order_update($order_id)
        {
            $booking = $this->get_booking_by_order_id($order_id);
            if (empty($booking['order_id'])) {
                return;
            }

            $this->settings = maybe_unserialize(get_post_meta($booking['form_id'], '_payamito_jet_booking', true)) ?: [];
            $this->general_settings = get_post_meta($booking['form_id'], 'payamito', true)['general'] ?? [];
            $this->booking = $booking;

            add_action('payamito/jet/booking/shutdown', [$this, 'pre_send']);
        }

        public function is_booking_status_change($response, $handler, $request)
        {
            if (strpos($request->get_route(), 'update-booking') !== false) {
                $params = $request->get_params()['item'] ?? [];
                $booking_id = absint($params['booking_id'] ?? 0);
                if ($booking_id) {
                    $booking = jet_abaf()->db->get_booking_by('booking_id', $booking_id);
                    $new_status = sanitize_text_field($params['status'] ?? '');
                    $old_status = $booking['status'] ?? '';
                    $request->set_param('is_status_changed', $new_status !== $old_status);
                }
            }
            return $response;
        }

        public function update($response, $handler, $request)
        {
            if (strpos($request->get_route(), 'update-booking') === false) {
                return $response;
            }

            $params = $request->get_params()['item'] ?? [];
            $booking_id = absint($params['booking_id'] ?? 0);
            if (!$booking_id) {
                return $response;
            }

            $booking = jet_abaf()->db->get_booking_by('booking_id', $booking_id);
            if (empty($booking['form_id']) || !$request->get_param('is_status_changed')) {
                return $response;
            }

            $this->settings = maybe_unserialize(get_post_meta($booking['form_id'], '_payamito_jet_booking', true)) ?: [];
            $this->general_settings = get_post_meta($booking['form_id'], 'payamito', true)['general'] ?? [];
            $this->booking = $booking;

            add_action('payamito/jet/booking/shutdown', [$this, 'pre_send']);
            return $response;
        }

        public function pre_process($pre, $booking, $apartment_booking)
        {
            Plugin::update_table();
            $form_id = absint($apartment_booking->getFormId());
            $request_data = $apartment_booking->getRequest();
            $this->settings = maybe_unserialize(get_post_meta($form_id, '_payamito_jet_booking', true)) ?: [];
            $this->general_settings = get_post_meta($form_id, 'payamito', true)['general'] ?? [];
            $user_field_name = sanitize_text_field($this->general_settings['user_phone_number'] ?? '');

            $this->admin_phones = array_map(function ($phone) {
                return ['phone_number' => sanitize_text_field($phone['phone_number'] ?? '')];
            }, (array) ($this->general_settings['admin_phone_number'] ?? []));

            $booking = array_merge($booking, $request_data);
            $booking['phone_number'] = sanitize_text_field($request_data[$user_field_name] ?? '');
            $booking['form_id'] = $form_id;
            $this->booking = $booking;

            return $pre;
        }

        public function add($booking_id)
        {
            $this->booking['booking_id'] = absint($booking_id);
            add_action('payamito/jet/booking/shutdown', [$this, 'pre_send']);
        }

        public function shutdown()
        {
            if (!empty($this->booking)) {
                do_action('payamito/jet/booking/shutdown', $this->booking);
            }
        }

        public function pre_send($booking)
        {
            if (empty($booking['form_id'])) {
                return;
            }

            $this->add_more_info($booking);

            $booking['check_in_date'] = date_i18n(get_option('date_format', 'H:i:s'), $booking['check_in_date'] ?? time());
            $booking['check_out_date'] = date_i18n(get_option('date_format', 'H:i:s'), $booking['check_out_date'] ?? time());
            $booking['apartment_title'] = get_the_title($booking['apartment_id'] ?? 0);
            $booking['status_title'] = \Payamito_Jet_Base::get_status_text($booking['status'] ?? '');

            if ($this->is_client_active('user') && $this->is_status_active('user', $booking['status'])) {
                $user = new Payamito_Jet_User($this->settings['user'][$booking['status']] ?? [], $booking);
                $user->apb_send();
            }

            if ($this->is_client_active('admin') && $this->is_status_active('admin', $booking['status'])) {
                $admin = new Payamito_Jet_Admin($this->settings['admin'][$booking['status']] ?? [], $booking);
                $admin->apb_send();
            }
        }

        public function add_more_info($booking)
        {
            global $wpdb;
            $table = $wpdb->prefix . 'jet_apartment_bookings';

            $phone = !empty($this->general_settings['enable_user_meta_phone_number'])
                ? get_user_meta($booking['user_id'] ?? 0, sanitize_text_field($this->general_settings['user_meta_phone_number'] ?? ''), true)
                : sanitize_text_field($booking['phone_number'] ?? '');

            $wpdb->update(
                $table,
                [
                    'form_id' => absint($booking['form_id'] ?? 0),
                    'phone_number' => $phone
                ],
                ['booking_id' => absint($booking['booking_id'] ?? 0)]
            );
        }

        public function is_client_active($type)
        {
            return !empty($this->settings[$type]['enable']);
        }

        public function is_status_active($client, $status)
        {
            return !empty($this->settings[$client][$status]['enable']);
        }

        public function filter_data($data)
        {
            $new_array = [];
            foreach ($data as $key => $value) {
                $new_array[ltrim($key, '_')] = $value;
            }
            $this->data = $new_array;
        }

        public function get_booking_by_order_id($order_id)
        {
            $booking = jet_abaf()->db->get_booking_by('order_id', absint($order_id));
            return $booking ?: [];
        }
    }
}